/*
 * Decompiled with CFR 0.152.
 */
package me.lucko.luckperms.common.backup;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.zip.GZIPOutputStream;
import me.lucko.luckperms.common.http.AbstractHttpClient;
import me.lucko.luckperms.common.http.UnsuccessfulRequestException;
import me.lucko.luckperms.common.locale.Message;
import me.lucko.luckperms.common.model.Group;
import me.lucko.luckperms.common.model.Track;
import me.lucko.luckperms.common.model.User;
import me.lucko.luckperms.common.node.utils.NodeJsonSerializer;
import me.lucko.luckperms.common.plugin.LuckPermsPlugin;
import me.lucko.luckperms.common.sender.Sender;
import me.lucko.luckperms.common.storage.Storage;
import me.lucko.luckperms.common.util.CompletableFutures;
import me.lucko.luckperms.common.util.gson.GsonProvider;
import me.lucko.luckperms.common.util.gson.JArray;
import me.lucko.luckperms.common.util.gson.JObject;
import me.lucko.luckperms.lib.adventure.text.Component;

public abstract class Exporter
implements Runnable {
    private static final SimpleDateFormat DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss z");
    protected final LuckPermsPlugin plugin;
    private final Sender executor;
    private final boolean includeUsers;
    private final boolean includeGroups;
    protected final ProgressLogger log;

    protected Exporter(LuckPermsPlugin plugin, Sender executor, boolean includeUsers, boolean includeGroups) {
        this.plugin = plugin;
        this.executor = executor;
        this.includeUsers = includeUsers;
        this.includeGroups = includeGroups;
        this.log = new ProgressLogger();
        this.log.addListener(plugin.getConsoleSender());
        this.log.addListener(executor);
    }

    @Override
    public void run() {
        JsonObject json = new JsonObject();
        json.add("metadata", (JsonElement)new JObject().add("generatedBy", this.executor.getNameWithLocation()).add("generatedAt", DATE_FORMAT.format(new Date(System.currentTimeMillis()))).toJson());
        if (this.includeGroups) {
            this.log.log("Gathering group data...");
            json.add("groups", (JsonElement)this.exportGroups());
            this.log.log("Gathering track data...");
            json.add("tracks", (JsonElement)this.exportTracks());
        }
        if (this.includeUsers) {
            this.log.log("Gathering user data...");
            json.add("users", (JsonElement)this.exportUsers());
        }
        this.processOutput(json);
    }

    protected abstract void processOutput(JsonObject var1);

    private JsonObject exportGroups() {
        JsonObject out = new JsonObject();
        List groups = this.plugin.getGroupManager().getAll().values().stream().sorted(Comparator.comparingInt(o -> o.getWeight().orElse(0)).reversed().thenComparing(Group::getName)).collect(Collectors.toList());
        for (Group group : groups) {
            out.add(group.getName(), (JsonElement)new JObject().add("nodes", (JsonElement)NodeJsonSerializer.serializeNodes(group.normalData().asSet())).toJson());
        }
        return out;
    }

    private JsonObject exportTracks() {
        JsonObject out = new JsonObject();
        Collection tracks = this.plugin.getTrackManager().getAll().values().stream().sorted(Comparator.comparing(Track::getName)).collect(Collectors.toList());
        for (Track track : tracks) {
            out.add(track.getName(), (JsonElement)new JObject().add("groups", new JArray().consume(arr -> track.getGroups().forEach(arr::add))).toJson());
        }
        return out;
    }

    private JsonObject exportUsers() {
        this.log.log("Finding a list of unique users to export.");
        Storage ds = this.plugin.getStorage();
        Set<UUID> users = ds.getUniqueUsers().join();
        this.log.log("Found " + users.size() + " unique users to export.");
        ExecutorService executor = Executors.newFixedThreadPool(32);
        HashSet<CompletableFuture<Void>> futures = new HashSet<CompletableFuture<Void>>();
        AtomicInteger userCount = new AtomicInteger(0);
        Map out = Collections.synchronizedMap(new TreeMap());
        for (UUID uuid : users) {
            futures.add(CompletableFuture.runAsync(() -> {
                User user = this.plugin.getStorage().loadUser(uuid, null).join();
                out.put(user.getUniqueId(), new JObject().consume(obj -> {
                    user.getUsername().ifPresent(username -> obj.add("username", (String)username));
                    if (!user.getPrimaryGroup().getStoredValue().orElse("default").equalsIgnoreCase("default")) {
                        obj.add("primaryGroup", user.getPrimaryGroup().getStoredValue().get());
                    }
                }).add("nodes", (JsonElement)NodeJsonSerializer.serializeNodes(user.normalData().asSet())).toJson());
                this.plugin.getUserManager().getHouseKeeper().cleanup(user.getUniqueId());
                userCount.incrementAndGet();
            }, executor));
        }
        CompletableFuture<Void> overallFuture = CompletableFutures.allOf(futures);
        while (true) {
            try {
                overallFuture.get(5L, TimeUnit.SECONDS);
            }
            catch (InterruptedException | ExecutionException e) {
                e.printStackTrace();
            }
            catch (TimeoutException e) {
                this.log.logProgress("Exported " + userCount.get() + " users so far.");
                continue;
            }
            break;
        }
        executor.shutdown();
        JsonObject outJson = new JsonObject();
        for (Map.Entry entry : out.entrySet()) {
            outJson.add(((UUID)entry.getKey()).toString(), (JsonElement)entry.getValue());
        }
        return outJson;
    }

    private static final class ProgressLogger {
        private final Set<Sender> listeners = new HashSet<Sender>();

        private ProgressLogger() {
        }

        public void addListener(Sender sender) {
            this.listeners.add(sender);
        }

        public Set<Sender> getListeners() {
            return this.listeners;
        }

        public void log(String msg) {
            this.dispatchMessage(Message.EXPORT_LOG, msg);
        }

        public void logProgress(String msg) {
            this.dispatchMessage(Message.EXPORT_LOG_PROGRESS, msg);
        }

        private void dispatchMessage(Message.Args1<String> messageType, String content) {
            Component message = messageType.build(content);
            for (Sender s : this.listeners) {
                s.sendMessage(message);
            }
        }
    }

    public static final class WebUpload
    extends Exporter {
        private final String label;

        public WebUpload(LuckPermsPlugin plugin, Sender executor, boolean includeUsers, boolean includeGroups, String label) {
            super(plugin, executor, includeUsers, includeGroups);
            this.label = label;
        }

        @Override
        protected void processOutput(JsonObject json) {
            this.log.log("Finished gathering data, uploading data...");
            ByteArrayOutputStream bytesOut = new ByteArrayOutputStream();
            try (OutputStreamWriter writer = new OutputStreamWriter((OutputStream)new GZIPOutputStream(bytesOut), StandardCharsets.UTF_8);){
                GsonProvider.normal().toJson((JsonElement)json, (Appendable)writer);
            }
            catch (IOException e) {
                this.plugin.getLogger().severe("Error compressing data", e);
            }
            try {
                String pasteId = this.plugin.getBytebin().postContent(bytesOut.toByteArray(), AbstractHttpClient.JSON_TYPE).key();
                this.log.getListeners().forEach(l -> Message.EXPORT_WEB_SUCCESS.send((Sender)l, pasteId, this.label));
            }
            catch (UnsuccessfulRequestException e) {
                this.log.getListeners().forEach(l -> Message.HTTP_REQUEST_FAILURE.send((Sender)l, e.getResponse().code(), e.getResponse().message()));
            }
            catch (IOException e) {
                this.plugin.getLogger().severe("Error uploading data to bytebin", e);
                this.log.getListeners().forEach(Message.HTTP_UNKNOWN_FAILURE::send);
            }
        }
    }

    public static final class SaveFile
    extends Exporter {
        private final Path filePath;

        public SaveFile(LuckPermsPlugin plugin, Sender executor, Path filePath, boolean includeUsers, boolean includeGroups) {
            super(plugin, executor, includeUsers, includeGroups);
            this.filePath = filePath;
        }

        @Override
        protected void processOutput(JsonObject json) {
            this.log.log("Finished gathering data, writing file...");
            try (BufferedWriter out = new BufferedWriter(new OutputStreamWriter((OutputStream)new GZIPOutputStream(Files.newOutputStream(this.filePath, new OpenOption[0])), StandardCharsets.UTF_8));){
                GsonProvider.normal().toJson((JsonElement)json, (Appendable)out);
            }
            catch (IOException e) {
                e.printStackTrace();
            }
            this.log.getListeners().forEach(l -> Message.EXPORT_FILE_SUCCESS.send((Sender)l, this.filePath.toFile().getAbsolutePath()));
        }
    }
}

